<?php
// register.php
// Page d'inscription de compte L2 pour Thanys

$pageTitle = 'Inscription';
require_once __DIR__ . '/config.php';

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// -------------------------------------------------
// 0) Statut serveurs (Login + Game) + nombre de joueurs
// -------------------------------------------------
$loginServerOnline = false;
$gameServerOnline  = false;
$onlineCount       = 0;

$ls = null;
$gs = null;

try {
    $ls = getLsPdo();              // Base LoginServer
    $ls->query('SELECT 1');
    $loginServerOnline = true;
} catch (Throwable $e) {
    $ls = null;
    $loginServerOnline = false;
}

try {
    $gs = getGsPdo();              // Base GameServer
    $gs->query('SELECT 1');
    $gameServerOnline = true;

    // Nombre de joueurs connectés (si la table existe)
    try {
        $onlineCount = (int)$gs->query("SELECT COUNT(*) FROM characters WHERE online = 1")->fetchColumn();
    } catch (Throwable $e2) {
        $onlineCount = 0;
    }
} catch (Throwable $e) {
    $gs = null;
    $gameServerOnline = false;
}

// -------------------------------------------------
// 1) Logique d'inscription
// -------------------------------------------------
$error   = '';
$success = '';

// Debug PHP (désactive en prod si besoin)
error_reporting(E_ALL);
ini_set('display_errors', 1);

/**
 * Hash du mot de passe pour L2J :
 * Base64( SHA1(password) en binaire )
 * -> même format que le loginserver (ex : m8NFsdVl2bKBH3gzSCsd74dpPw=)
 */
function l2_hash_password(string $password): string
{
    return base64_encode(sha1($password, true));
}

// Traitement du formulaire
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $login    = trim($_POST['login']    ?? '');
    $password = $_POST['password']      ?? '';
    $confirm  = $_POST['password2']     ?? '';
    $email    = trim($_POST['email']    ?? '');

    // 1) Vérifs de base
    if ($login === '' || $password === '' || $confirm === '' || $email === '') {
        $error = "Tous les champs marqués sont obligatoires.";
    } elseif ($password !== $confirm) {
        $error = "Les mots de passe ne correspondent pas.";
    } else {
        // 2) Vérifier que login / password respectent les templates L2
        if (!preg_match('/^[A-Za-z0-9]{4,14}$/', $login)) {
            $error = "Le nom de compte doit contenir 4 à 14 caractères alphanumériques (A-Z, a-z, 0-9).";
        } elseif (!preg_match('/^[A-Za-z0-9]{4,16}$/', $password)) {
            $error = "Le mot de passe doit contenir 4 à 16 caractères alphanumériques (A-Z, a-z, 0-9).";
        } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $error = "L'adresse e-mail n'est pas valide.";
        }
    }

    if ($error === '') {
        // 3) Vérifier qu'on peut contacter le LoginServer
        if (!$ls) {
            $error = "Impossible de contacter le serveur de comptes. Veuillez réessayer un peu plus tard.";
        } else {
            try {
                // Vérifier que le login n'existe pas déjà
                $stmt = $ls->prepare("SELECT login FROM accounts WHERE login = :l LIMIT 1");
                $stmt->execute([':l' => $login]);
                if ($stmt->fetch()) {
                    $error = "Ce nom de compte est déjà utilisé.";
                } else {
                    // Vérifier unicité de l'e-mail
                    $stmt = $ls->prepare("SELECT email FROM accounts WHERE email = :e LIMIT 1");
                    $stmt->execute([':e' => $email]);
                    if ($stmt->fetch()) {
                        $error = "Cette adresse e-mail est déjà associée à un compte.";
                    }
                }

                if ($error === '') {
                    // Hasher le mot de passe au format attendu par le loginserver
                    $gamePassword = l2_hash_password($password);

                    // Insérer le compte dans la table accounts (LoginServer)
                    $stmt = $ls->prepare("
                        INSERT INTO accounts (
                            login,
                            password,
                            email,
                            created_time,
                            lastactive,
                            accessLevel
                        ) VALUES (
                            :l,
                            :p,
                            :e,
                            NOW(),
                            0,
                            0
                        )
                    ");

                    $stmt->execute([
                        ':l' => $login,
                        ':p' => $gamePassword,
                        ':e' => $email,
                    ]);

                    $success = "Compte créé avec succès ! Vous pouvez maintenant vous connecter en jeu.";
                }
            } catch (Throwable $e) {
                $error = "Erreur de connexion à la base de comptes : " . htmlspecialchars($e->getMessage());
            }
        }
    }
}

// -------------------------------------------------
// 2) Affichage (header + contenu + footer)
// -------------------------------------------------
include __DIR__ . '/includes/header.php';
?>

<style>
    /* Mise en page spécifique au formulaire d'inscription */
    .register-card {
        margin-top: 1.3rem;
    }

    .register-form {
        margin-top: 0.8rem;
    }

    .register-fields-row {
        display: flex;
        flex-wrap: wrap;
        gap: 0.9rem 1.1rem;
        align-items: flex-end;
    }

    .register-field {
        flex: 1 1 210px;
        min-width: 200px;
        font-size: 0.9rem;
    }

    .register-field label {
        display: block;
        font-size: 0.86rem;
        margin-bottom: 0.18rem;
    }

    .register-field input[type="text"],
    .register-field input[type="password"],
    .register-field input[type="email"] {
        width: 100%;
        padding: 0.45rem 0.55rem;
        border-radius: 10px;
        border: 1px solid #1f2937;
        background: #020617;
        color: #e5e7eb;
        font-size: 0.9rem;
    }

    .register-submit {
        margin-top: 0.9rem;
    }

    .register-submit button {
        width: 100%;
        padding: 0.55rem 0.9rem;
        border-radius: 999px;
        border: none;
        background: linear-gradient(90deg, #22c1c3, #2563eb);
        color: #020617;
        font-weight: 600;
        font-size: 0.9rem;
        cursor: pointer;
    }

    .register-submit button:hover {
        filter: brightness(1.05);
    }

    .register-hint {
        margin-top: 0.7rem;
        font-size: 0.86rem;
        color: #9ca3af;
    }

    .alert-error {
        padding: 0.5rem 0.7rem;
        border-radius: 0.7rem;
        background: rgba(248,113,113,0.08);
        border: 1px solid rgba(248,113,113,0.7);
        color: #fecaca;
        font-size: 0.85rem;
        margin-bottom: 0.8rem;
    }

    .alert-success {
        padding: 0.5rem 0.7rem;
        border-radius: 0.7rem;
        background: rgba(34,197,94,0.08);
        border: 1px solid rgba(34,197,94,0.7);
        color: #bbf7d0;
        font-size: 0.85rem;
        margin-bottom: 0.8rem;
    }

    @media (max-width: 900px) {
        .register-fields-row {
            flex-direction: column;
        }
        .register-field {
            min-width: 100%;
        }
    }
</style>

<div class="card register-card">
    <h1 class="page-title">Création de compte</h1>
    <p class="page-intro">
        Créez votre compte pour vous connecter sur <?= htmlspecialchars($SERVER_NAME) ?>.
    </p>

    <!-- Statut serveur sous le header de la carte -->
    <div class="stat-grid" style="margin:0.5rem 0 1rem 0;">
        <div class="stat-card">
            <div class="stat-label">Login server</div>
            <div class="stat-value" style="color:<?= $loginServerOnline ? '#4ade80' : '#f97373' ?>;">
                <?= $loginServerOnline ? 'En ligne' : 'Hors ligne' ?>
            </div>
        </div>

        <div class="stat-card">
            <div class="stat-label">Game server</div>
            <div class="stat-value" style="color:<?= $gameServerOnline ? '#4ade80' : '#f97373' ?>;">
                <?= $gameServerOnline ? 'En ligne' : 'Hors ligne' ?>
            </div>
        </div>

        <div class="stat-card">
            <div class="stat-label">Joueurs connectés</div>
            <div class="stat-value"><?= (int)$onlineCount ?></div>
        </div>
    </div>

    <?php if ($error): ?>
        <div class="alert-error"><?= htmlspecialchars($error) ?></div>
    <?php endif; ?>

    <?php if ($success): ?>
        <div class="alert-success"><?= htmlspecialchars($success) ?></div>
    <?php endif; ?>

    <form method="post" action="register.php" class="register-form">
        <div class="register-fields-row">
            <div class="register-field">
                <label for="login">Nom de compte *</label>
                <input
                    type="text"
                    name="login"
                    id="login"
                    required
                    maxlength="14"
                    value="<?= isset($login) ? htmlspecialchars($login, ENT_QUOTES, 'UTF-8') : '' ?>"
                >
            </div>

            <div class="register-field">
                <label for="password">Mot de passe *</label>
                <input
                    type="password"
                    name="password"
                    id="password"
                    required
                    maxlength="16"
                >
            </div>

            <div class="register-field">
                <label for="password2">Confirmation du mot de passe *</label>
                <input
                    type="password"
                    name="password2"
                    id="password2"
                    required
                    maxlength="16"
                >
            </div>

            <div class="register-field">
                <label for="email">Adresse e-mail *</label>
                <input
                    type="email"
                    name="email"
                    id="email"
                    required
                    value="<?= isset($email) ? htmlspecialchars($email, ENT_QUOTES, 'UTF-8') : '' ?>"
                >
            </div>
        </div>

        <div class="register-submit">
            <button type="submit">Créer mon compte</button>
        </div>
    </form>

    <div class="register-hint">
        Login : 4–14 caractères A–Z, a–z, 0–9<br>
        Mot de passe : 4–16 caractères A–Z, a–z, 0–9
    </div>
</div>

<?php include __DIR__ . '/includes/footer.php'; ?>

