<?php
// quests.php – Liste des quêtes (version simplifiée, sans termes techniques)

$pageTitle = 'Quêtes';
require_once 'config.php';

$gs = getGsPdo(); // base GameServer

$search       = trim($_GET['q'] ?? '');
$selectedName = isset($_GET['quest']) ? (string)$_GET['quest'] : '';

$quests        = [];
$selectedQuest = [];
$error         = null;

try {
    // Liste des quêtes (groupées par quest_name)
    if ($search !== '') {
        $stmt = $gs->prepare("
            SELECT quest_name, COUNT(*) AS vars_count
            FROM quest_global_data
            WHERE quest_name LIKE :term
               OR value      LIKE :term
            GROUP BY quest_name
            ORDER BY quest_name ASC
        ");
        $stmt->execute([':term' => '%' . $search . '%']);
    } else {
        $stmt = $gs->query("
            SELECT quest_name, COUNT(*) AS vars_count
            FROM quest_global_data
            GROUP BY quest_name
            ORDER BY quest_name ASC
        ");
    }
    $quests = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Détail d'une quête sélectionnée
    if ($selectedName !== '') {
        $stmt2 = $gs->prepare("
            SELECT var, value
            FROM quest_global_data
            WHERE quest_name = :qname
            ORDER BY var ASC
        ");
        $stmt2->execute([':qname' => $selectedName]);
        $selectedQuest = $stmt2->fetchAll(PDO::FETCH_ASSOC);
    }
} catch (Throwable $e) {
    $error = "Une erreur est survenue lors du chargement des quêtes.";
}

include 'includes/header.php';
?>

<!-- Bandeau visuel -->
<div class="card">
    <div style="border-radius:1rem;overflow:hidden;border:1px solid #1f2937;">
        <!-- Mets ici l'image de bannière que tu veux -->
        <div style="
            width:100%;
            height:220px;
            background:url('assets/hf-quest-banner.jpg') center/cover no-repeat;
        "></div>
    </div>
</div>

<div class="card">
    <h1 class="page-title">Quêtes</h1>
    <p class="page-intro">
        Parcours l'ensemble des quêtes disponibles sur le serveur et consulte leurs informations.
    </p>

    <?php if ($error): ?>
        <p style="color:#f97373;"><?= htmlspecialchars($error) ?></p>
    <?php endif; ?>

    <!-- Recherche -->
    <form method="get" style="margin:0.8rem 0;display:flex;gap:0.5rem;flex-wrap:wrap;align-items:center;">
        <input
            type="text"
            name="q"
            value="<?= htmlspecialchars($search) ?>"
            placeholder="Rechercher une quête par nom ou mot-clé..."
            style="
                flex:1 1 220px;
                padding:0.4rem 0.6rem;
                border-radius:999px;
                border:1px solid #1f2937;
                background:#020617;
                color:#e5e7eb;
                font-size:0.85rem;
            "
        >
        <?php if ($selectedName !== ''): ?>
            <input type="hidden" name="quest" value="<?= htmlspecialchars($selectedName) ?>">
        <?php endif; ?>
        <button type="submit"
            style="
                padding:0.4rem 0.9rem;
                border-radius:999px;
                border:none;
                cursor:pointer;
                font-size:0.85rem;
                font-weight:600;
                background:linear-gradient(135deg,#38bdf8,#1d4ed8);
                color:#020617;
            ">
            Rechercher
        </button>
    </form>

    <!-- Deux colonnes : liste + détail -->
    <div style="display:flex;gap:1rem;align-items:flex-start;flex-wrap:wrap;margin-top:0.4rem;">

        <!-- Colonne gauche : liste des quêtes -->
        <div style="flex:1 1 260px;min-width:260px;">
            <h2 style="font-size:0.9rem;margin:0.2rem 0 0.4rem;">Liste des quêtes</h2>

            <?php if (empty($quests)): ?>
                <p class="muted">Aucune quête n'a encore été enregistrée.</p>
            <?php else: ?>
                <ul style="list-style:none;padding:0;margin:0;max-height:420px;overflow:auto;">
                    <?php foreach ($quests as $q): ?>
                        <?php
                        $qName      = $q['quest_name'];
                        $isSelected = ($selectedName !== '' && $selectedName === $qName);
                        // Nom lisible : on remplace les _ par des espaces
                        $prettyName = ucwords(str_replace('_', ' ', $qName));
                        ?>
                        <li style="margin-bottom:0.25rem;">
                            <a href="quests.php?quest=<?= urlencode($qName) ?>&amp;q=<?= urlencode($search) ?>"
                               style="
                                    display:flex;
                                    flex-direction:column;
                                    padding:0.35rem 0.55rem;
                                    border-radius:0.75rem;
                                    text-decoration:none;
                                    background:<?= $isSelected ? 'rgba(37,99,235,0.22)' : 'rgba(15,23,42,0.95)' ?>;
                                    border:1px solid <?= $isSelected ? '#3b82f6' : '#1f2937' ?>;
                                    color:#e5e7eb;
                                    font-size:0.83rem;
                               ">
                                <span style="font-weight:500;">
                                    <?= htmlspecialchars($prettyName) ?>
                                </span>
                                <!-- Sous-ligne neutre, sans termes techniques -->
                                <span style="font-size:0.75rem;color:#9ca3af;margin-top:0.15rem;">
                                    Informations disponibles : <?= (int)($q['vars_count'] ?? 0) ?>
                                </span>
                            </a>
                        </li>
                    <?php endforeach; ?>
                </ul>
            <?php endif; ?>
        </div>

        <!-- Colonne droite : détail d'une quête -->
        <div style="flex:2 1 340px;min-width:300px;">
            <h2 style="font-size:0.9rem;margin:0.2rem 0 0.4rem;">Détail de la quête</h2>

            <?php if ($selectedName === '' || empty($selectedQuest)): ?>
                <p class="muted">
                    Sélectionne une quête dans la liste de gauche pour afficher ses informations.
                </p>
            <?php else: ?>
                <?php
                $prettyName = ucwords(str_replace('_', ' ', $selectedName));

                // Image automatique si dispo : assets/quests/<quest_name>.(jpg|png|jpeg|webp)
                $questImageWeb = null;
                $extensions    = ['jpg','png','jpeg','webp'];
                foreach ($extensions as $ext) {
                    $diskPath = __DIR__ . '/assets/quests/' . $selectedName . '.' . $ext;
                    if (file_exists($diskPath)) {
                        $questImageWeb = 'assets/quests/' . rawurlencode($selectedName) . '.' . $ext;
                        break;
                    }
                }
                ?>

                <h3 style="font-size:1rem;margin:0.2rem 0 0.4rem;">
                    <?= htmlspecialchars($prettyName) ?>
                </h3>

                <?php if ($questImageWeb): ?>
                    <div style="border-radius:0.9rem;overflow:hidden;border:1px solid #1f2937;margin-bottom:0.6rem;">
                        <img src="<?= $questImageWeb ?>"
                             alt="Illustration de <?= htmlspecialchars($prettyName) ?>"
                             style="display:block;width:100%;max-height:260px;object-fit:cover;">
                    </div>
                <?php endif; ?>

                <p class="muted" style="margin-bottom:0.4rem;">
                    Résumé des informations disponibles pour cette quête.
                </p>

                <table style="width:100%;border-collapse:collapse;font-size:0.84rem;margin-top:0.2rem;">
                    <thead>
                        <tr style="border-bottom:1px solid #1f2937;">
                            <th style="text-align:left;padding:0.35rem 0.4rem;color:#9ca3af;width:30%;">Information</th>
                            <th style="text-align:left;padding:0.35rem 0.4rem;color:#9ca3af;">Détail</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($selectedQuest as $row): ?>
                            <tr style="border-bottom:1px solid #111827;">
                                <td style="padding:0.3rem 0.4rem;"><?= htmlspecialchars($row['var']) ?></td>
                                <td style="padding:0.3rem 0.4rem;"><?= htmlspecialchars($row['value']) ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
    </div>
</div>

<?php include 'includes/footer.php'; ?>


