<?php
$pageTitle = 'Comptes & joueurs en ligne';
require_once 'config.php';

$ls = getLsPdo();   // Base LoginServer
$gs = getGsPdo();   // Base GameServer

// -------------------------------------------------
// 1) Statut des serveurs (basé sur l'accès DB)
// -------------------------------------------------
$loginServerOnline = false;
$gameServerOnline  = false;

try {
    $ls->query('SELECT 1');
    $loginServerOnline = true;
} catch (Throwable $e) {
    $loginServerOnline = false;
}

try {
    $gs->query('SELECT 1');
    $gameServerOnline = true;
} catch (Throwable $e) {
    $gameServerOnline = false;
}

// -------------------------------------------------
// 2) Comptes LoginServer + infos personnages
// -------------------------------------------------
$accounts       = [];
$errorAccounts  = null;

try {
    // 1) Récupération de la liste des logins côté LoginServer
    $stmt = $ls->query("
        SELECT login
        FROM accounts
        ORDER BY login ASC
    ");
    $logins = $stmt->fetchAll(PDO::FETCH_COLUMN);

    if (!empty($logins)) {
        // Initialiser le tableau des comptes avec les colonnes supplémentaires vides
        foreach ($logins as $login) {
            $accounts[] = [
                'login'               => $login,
                'char_name'           => null,
                'clan_name'           => null,
                'lastAccessFormatted' => null,
            ];
        }

        // 2) Récupérer les personnages associés dans la base GameServer
        //    On prend l'info perso + clan + dernière connexion.
        $placeholders = implode(',', array_fill(0, count($logins), '?'));

        $sqlChars = "
            SELECT
                c.account_name,
                c.char_name,
                c.lastAccess,
                cl.clan_name
            FROM characters AS c
            LEFT JOIN clan_data AS cl ON cl.clan_id = c.clanid
            WHERE c.account_name IN ($placeholders)
            ORDER BY c.level DESC, c.lastAccess DESC
        ";

        $stmtChars = $gs->prepare($sqlChars);
        $stmtChars->execute($logins);

        $charsByAccount = [];

        while ($row = $stmtChars->fetch(PDO::FETCH_ASSOC)) {
            $accName = $row['account_name'] ?? '';
            if ($accName === '') {
                continue;
            }

            // Normalisation en minuscule pour éviter les soucis de casse
            $key = strtolower($accName);

            // On ne garde que le premier résultat par compte (considéré comme "perso principal")
            if (!isset($charsByAccount[$key])) {
                $last      = $row['lastAccess'] ?? null;
                $formatted = null;

                if (!empty($last) && $last != 0) {
                    $ts = (int)$last;

                    // Beaucoup de L2J stockent en millisecondes -> on adapte
                    if ($ts > 2000000000) { // trop grand pour des secondes
                        $ts = (int)($ts / 1000);
                    }
                    if ($ts > 0) {
                        $formatted = date('Y-m-d H:i', $ts);
                    }
                }

                $charsByAccount[$key] = [
                    'char_name'           => $row['char_name'] ?? null,
                    'clan_name'           => $row['clan_name'] ?? null,
                    'lastAccessFormatted' => $formatted,
                ];
            }
        }

        // 3) Fusionner les infos perso/clan/date dans le tableau des comptes
        foreach ($accounts as &$acc) {
            $key  = strtolower($acc['login']);      // même normalisation ici
            $info = $charsByAccount[$key] ?? null;

            $acc['char_name']           = $info['char_name'] ?? null;
            $acc['clan_name']           = $info['clan_name'] ?? null;
            $acc['lastAccessFormatted'] = $info['lastAccessFormatted'] ?? null;
        }
        unset($acc);
    }
} catch (Throwable $e) {
    $errorAccounts = $e->getMessage();
}

// -------------------------------------------------
// 3) Joueurs en ligne
// -------------------------------------------------
$onlineCount   = 0;
$players       = [];
$errorPlayers  = null;

try {
    $onlineCount = (int)$gs->query("SELECT COUNT(*) FROM characters WHERE online = 1")->fetchColumn();

    $stmt = $gs->query("
        SELECT char_name, level
        FROM characters
        WHERE online = 1
        ORDER BY level DESC, char_name ASC
        LIMIT 200
    ");
    $players = $stmt->fetchAll();
} catch (Throwable $e) {
    $errorPlayers = $e->getMessage();
}

include 'includes/header.php';
?>

<!-- 4) Statut serveurs (sous le header) -->
<div class="card">
    <h1 class="page-title">Statut des serveurs</h1>
    <p class="page-intro">
        État du LoginServer et du GameServer de <?= htmlspecialchars($SERVER_NAME) ?> (sans afficher les adresses IP).
    </p>

    <div class="stat-grid" style="margin-top:0.5rem;">
        <div class="stat-card">
            <div class="stat-label">Login server</div>
            <div class="stat-value" style="color:<?= $loginServerOnline ? '#4ade80' : '#f97373' ?>;">
                <?= $loginServerOnline ? 'En ligne' : 'Hors ligne' ?>
            </div>
        </div>

        <div class="stat-card">
            <div class="stat-label">Game server</div>
            <div class="stat-value" style="color:<?= $gameServerOnline ? '#4ade80' : '#f97373' ?>;">
                <?= $gameServerOnline ? 'En ligne' : 'Hors ligne' ?>
            </div>
        </div>

        <div class="stat-card">
            <div class="stat-label">Joueurs connectés</div>
            <div class="stat-value"><?= (int)$onlineCount ?></div>
        </div>
    </div>
</div>

<!-- 5) Deux colonnes : comptes + joueurs, toujours visibles sur la même page -->
<div class="layout-grid"
     style="display:grid;grid-template-columns:minmax(0,1.2fr) minmax(0,1fr);gap:1.5rem;align-items:flex-start;margin-top:1rem;">

    <!-- Colonne gauche : comptes -->
    <div class="card">
        <h2 class="page-title">Comptes LoginServer</h2>
        <p class="page-intro">
            Liste des comptes enregistrés côté LoginServer pour <?= htmlspecialchars($SERVER_NAME) ?>.
        </p>

        <?php if ($errorAccounts): ?>
            <p style="color:#f97373;">Erreur lors de la lecture des comptes : <?= htmlspecialchars($errorAccounts) ?></p>
        <?php elseif (empty($accounts)): ?>
            <p class="muted">Aucun compte trouvé pour le moment.</p>
        <?php else: ?>
            <!-- Scroll interne pour garder la carte visible -->
            <div style="max-height:600px;overflow:auto;margin-top:0.5rem;">
                <table style="width:100%;border-collapse:collapse;font-size:0.9rem;">
                    <thead>
                        <tr style="border-bottom:1px solid #1f2937;">
                            <th style="text-align:left;padding:0.35rem 0.4rem;color:#9ca3af;">Login</th>
                            <th style="text-align:left;padding:0.35rem 0.4rem;color:#9ca3af;">Personnage principal</th>
                            <th style="text-align:left;padding:0.35rem 0.4rem;color:#9ca3af;">Guilde</th>
                            <th style="text-align:left;padding:0.35rem 0.4rem;color:#9ca3af;">Dernière connexion</th>
                        </tr>
                    </thead>
                    <tbody>
                    <?php foreach ($accounts as $acc): ?>
                        <tr style="border-bottom:1px solid #111827;">
                            <td style="padding:0.3rem 0.4rem;"><?= htmlspecialchars($acc['login']) ?></td>
                            <td style="padding:0.3rem 0.4rem;">
                                <?= $acc['char_name'] ? htmlspecialchars($acc['char_name']) : '—' ?>
                            </td>
                            <td style="padding:0.3rem 0.4rem;">
                                <?= $acc['clan_name'] ? htmlspecialchars($acc['clan_name']) : '—' ?>
                            </td>
                            <td style="padding:0.3rem 0.4rem;">
                                <?= $acc['lastAccessFormatted'] ? htmlspecialchars($acc['lastAccessFormatted']) : '—' ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
    </div>

    <!-- Colonne droite : joueurs en ligne -->
    <div class="card">
        <h2 class="page-title">Joueurs en ligne</h2>
        <p class="page-intro">
            Personnages actuellement connectés sur l'univers <?= htmlspecialchars($SERVER_NAME) ?>.
        </p>

        <?php if ($errorPlayers): ?>
            <p style="color:#f97373;">Erreur lors de la lecture des joueurs : <?= htmlspecialchars($errorPlayers) ?></p>
        <?php elseif ($onlineCount <= 0 || empty($players)): ?>
            <p class="muted">
                Aucun joueur n'est connecté pour le moment. Revenez un peu plus tard,
                ou rassemblez vos amis pour envahir Aden !
            </p>
        <?php else: ?>
            <!-- Scroll interne pour garder la carte visible -->
            <div style="max-height:600px;overflow:auto;margin-top:0.5rem;">
                <table style="width:100%;border-collapse:collapse;font-size:0.9rem;">
                    <thead>
                        <tr style="border-bottom:1px solid #1f2937;">
                            <th style="text-align:left;padding:0.35rem 0.4rem;color:#9ca3af;">Nom</th>
                            <th style="text-align:left;padding:0.35rem 0.4rem;color:#9ca3af;">Niveau</th>
                        </tr>
                    </thead>
                    <tbody>
                    <?php foreach ($players as $p): ?>
                        <tr style="border-bottom:1px solid #111827;">
                            <td style="padding:0.3rem 0.4rem;"><?= htmlspecialchars($p['char_name']) ?></td>
                            <td style="padding:0.3rem 0.4rem;"><?= (int)$p['level'] ?></td>
                        </tr>
                    <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
    </div>
</div>

<?php include 'includes/footer.php'; ?>
