<?php
// maps.php – Cartes & zones (version visuelle avec menu déroulant de zones)

$pageTitle = 'Cartes & zones';
require_once 'config.php';

$serverName = isset($SERVER_NAME) ? $SERVER_NAME : 'Thanys';

/**
 * Définition des maps et zones.
 * Tu peux modifier / compléter ces données comme tu veux.
 * Pense à déposer les images dans assets/maps/
 */
$maps = [
    [
        'id' => 'world',
        'title' => 'Carte du monde',
        'description' => 'Vue globale des principales régions du monde ' . $serverName . '.',
        'image' => 'assets/maps/worldmap.jpg', // image à créer / adapter
        'zones' => [
            [
                'name'  => 'Talking Island',
                'level' => '1 – 20',
                'type'  => 'Zone de départ',
                'mobs'  => 'Loups, Gremlins, Orcs débutants…',
                'dungeons' => 'Caverne d\'épreuve (exemple)',
                'notes' => 'Idéale pour les premières quêtes et la découverte du serveur.'
            ],
            [
                'name'  => 'Gludio Territory',
                'level' => '15 – 35',
                'type'  => 'Zone d\'XP',
                'mobs'  => 'Orcs, gobelins, bandits des routes…',
                'dungeons' => 'Ruines de Gludio, premiers spots de groupe.',
                'notes' => 'Transition entre la zone de départ et les régions plus dangereuses.'
            ],
            [
                'name'  => 'Dion Territory',
                'level' => '25 – 40',
                'type'  => 'Zone d\'XP',
                'mobs'  => 'Monstres des marais, créatures corrompues…',
                'dungeons' => 'Cruma Marshlands, accès à la Tour de Cruma.',
                'notes' => 'Beaucoup de quêtes pour personnages intermédiaires.'
            ],
            [
                'name'  => 'Giran Territory',
                'level' => '35 – 60',
                'type'  => 'Région d’XP & commerce',
                'mobs'  => 'Monstres des plaines, créatures draconiques en périphérie.',
                'dungeons' => 'Dragon Valley, Lair of Antharas (HL).',
                'notes' => 'Zone très fréquentée, hub important pour le mid / high level.'
            ],
        ],
    ],
    [
        'id' => 'ti_gludio',
        'title' => 'Talking Island & Gludio',
        'description' => 'Zones de départ et premières régions à explorer.',
        'image' => 'assets/maps/ti_gludio.jpg',
        'zones' => [
            [
                'name'  => 'Talking Island Village',
                'level' => '1 – 10',
                'type'  => 'Village de départ',
                'mobs'  => 'Très peu de monstres autour du village.',
                'dungeons' => '',
                'notes' => 'Lieu idéal pour débuter, gérer les premières quêtes et tutoriels.'
            ],
            [
                'name'  => 'Talking Island Fields',
                'level' => '5 – 15',
                'type'  => 'Zone d\'XP',
                'mobs'  => 'Loups, Gremlins, araignées, petits orcs.',
                'dungeons' => 'Petites cavernes locales (exemple).',
                'notes' => 'Parfait pour les joueurs solo juste après la création du personnage.'
            ],
            [
                'name'  => 'Gludin Village',
                'level' => '15 – 25',
                'type'  => 'Village & carrefour',
                'mobs'  => '',
                'dungeons' => 'Accès rapide à plusieurs zones d’XP environnantes.',
                'notes' => 'Point de passage entre les premières zones et le reste du continent.'
            ],
            [
                'name'  => 'Orc Barracks',
                'level' => '20 – 30',
                'type'  => 'Zone d\'XP',
                'mobs'  => 'Orcs guerriers, archers, chefs orcs.',
                'dungeons' => '',
                'notes' => 'Zone extérieure adaptée aux groupes petits ou moyens.'
            ],
            [
                'name'  => 'Abandoned Camp',
                'level' => '20 – 30',
                'type'  => 'Zone d\'XP',
                'mobs'  => 'Bandits, soldats déserteurs…',
                'dungeons' => '',
                'notes' => 'Zone assez compacte, bonne XP pour les mages et archers.'
            ],
        ],
    ],
    [
        'id' => 'dion',
        'title' => 'Région de Dion',
        'description' => 'Région marécageuse et champs de bataille avec accès à Cruma.',
        'image' => 'assets/maps/dion_region.jpg',
        'zones' => [
            [
                'name'  => 'Dion Town',
                'level' => '20 – 30',
                'type'  => 'Ville',
                'mobs'  => '',
                'dungeons' => 'Point de départ vers les marécages et la Tour de Cruma.',
                'notes' => 'Ville centrale pour les quêtes de niveau intermédiaire.'
            ],
            [
                'name'  => 'Floran Agricultural Area',
                'level' => '25 – 35',
                'type'  => 'Zone d\'XP',
                'mobs'  => 'Créatures agricoles corrompues, bandits.',
                'dungeons' => '',
                'notes' => 'Zone assez ouverte, facile pour XP en solo ou petit groupe.'
            ],
            [
                'name'  => 'Cruma Marshlands',
                'level' => '30 – 40',
                'type'  => 'Zone d\'XP',
                'mobs'  => 'Monstres des marais, insectes géants, golems…',
                'dungeons' => 'Entrée de la Tour de Cruma.',
                'notes' => 'Préparation idéale avant d’entrer dans la Tour de Cruma.'
            ],
            [
                'name'  => 'Cruma Tower',
                'level' => '35 – 52',
                'type'  => 'Donjon',
                'mobs'  => 'Golems, monstres magiques, gardiens de la Tour.',
                'dungeons' => 'Donjon multi-étages avec plusieurs spots pour groupes.',
                'notes' => 'Donjon clé pour XP mid-level, très fréquenté en groupe.'
            ],
        ],
    ],
    [
        'id' => 'giran',
        'title' => 'Région de Giran',
        'description' => 'Région commerciale majeure avec zones d’XP inter et haut niveau.',
        'image' => 'assets/maps/giran_region.jpg',
        'zones' => [
            [
                'name'  => 'Giran Town',
                'level' => '30 – 40',
                'type'  => 'Ville / hub',
                'mobs'  => '',
                'dungeons' => 'Accès à Dragon Valley, Execution Grounds, etc.',
                'notes' => 'Hub économique majeur, souvent très peuplé.'
            ],
            [
                'name'  => 'Execution Grounds',
                'level' => '30 – 40',
                'type'  => 'Zone d\'XP',
                'mobs'  => 'Squelettes, zombies, esprits vengeurs…',
                'dungeons' => '',
                'notes' => 'Bon spot pour groupes de niveau intermédiaire.'
            ],
            [
                'name'  => 'Dragon Valley',
                'level' => '60 – 75',
                'type'  => 'Zone d\'XP haut niveau',
                'mobs'  => 'Dragons, drakes, créatures draconiques.',
                'dungeons' => 'Accès vers la Lair of Antharas.',
                'notes' => 'Zone dangereuse, excellente XP mais mobs très agressifs.'
            ],
            [
                'name'  => 'Lair of Antharas',
                'level' => 'Raid HL',
                'type'  => 'Zone de raid',
                'mobs'  => 'Antharas et ses gardiens.',
                'dungeons' => 'Instance / zone de raid majeure.',
                'notes' => 'Réservée aux alliances / clans bien organisés.'
            ],
        ],
    ],
    [
        'id' => 'oren',
        'title' => 'Région d’Oren',
        'description' => 'Région montagneuse et boisée, avec plusieurs donjons.',
        'image' => 'assets/maps/oren_region.jpg',
        'zones' => [
            [
                'name'  => 'Oren Town',
                'level' => '40 – 50',
                'type'  => 'Ville',
                'mobs'  => '',
                'dungeons' => 'Point de départ vers Enchanted Valley et Sea of Spores.',
                'notes' => 'Ville intermédiaire avec accès aux spots montagnards.'
            ],
            [
                'name'  => 'Enchanted Valley',
                'level' => '45 – 55',
                'type'  => 'Zone d\'XP',
                'mobs'  => 'Fées, dryades, créatures magiques.',
                'dungeons' => '',
                'notes' => 'Zone très jolie visuellement, bons spots pour groupes.'
            ],
            [
                'name'  => 'Sea of Spores',
                'level' => '55 – 65',
                'type'  => 'Zone dangereuse',
                'mobs'  => 'Insectes géants, spores toxiques, monstres corrompus.',
                'dungeons' => '',
                'notes' => 'Zone HL, dégâts magiques élevés, à faire en groupe bien préparé.'
            ],
        ],
    ],
    [
        'id' => 'aden',
        'title' => 'Région d’Aden',
        'description' => 'Région centrale avec plaines, forêts maudites et zones HL.',
        'image' => 'assets/maps/aden_region.jpg',
        'zones' => [
            [
                'name'  => 'Aden Castle Town',
                'level' => '50 – 60',
                'type'  => 'Ville',
                'mobs'  => '',
                'dungeons' => 'Point de départ vers Forest of Mirrors, Fields of Massacre…',
                'notes' => 'Ville centrale pour les niveaux 60+.'
            ],
            [
                'name'  => 'Forest of Mirrors',
                'level' => '58 – 65',
                'type'  => 'Zone dangereuse',
                'mobs'  => 'Esprits, créatures illusoires, monstres reflétés.',
                'dungeons' => '',
                'notes' => 'Zone complexe, nécessite un bon groupe organisé.'
            ],
            [
                'name'  => 'Fields of Massacre',
                'level' => '65 – 75',
                'type'  => 'Zone d\'XP HL',
                'mobs'  => 'Soldats morts-vivants, monstres de guerre…',
                'dungeons' => '',
                'notes' => 'Zone très agressive, idéale pour les groupes HL.'
            ],
        ],
    ],
    [
        'id' => 'rune_goddard',
        'title' => 'Régions de Rune & Goddard',
        'description' => 'Régions nordiques et volcaniques, très haut niveau.',
        'image' => 'assets/maps/rune_goddard.jpg',
        'zones' => [
            [
                'name'  => 'Rune Castle Town',
                'level' => '70+',
                'type'  => 'Ville HL',
                'mobs'  => '',
                'dungeons' => 'Accès à Forest of the Dead, Valley of Saints…',
                'notes' => 'Ville pour les joueurs de très haut niveau.'
            ],
            [
                'name'  => 'Forest of the Dead',
                'level' => '70 – 78',
                'type'  => 'Zone dangereuse',
                'mobs'  => 'Esprits morts, fantômes, morts-vivants puissants.',
                'dungeons' => '',
                'notes' => 'Zone très sombre, nécessite un groupe solide.'
            ],
            [
                'name'  => 'Valley of Saints',
                'level' => '75 – 80',
                'type'  => 'Zone d\'XP HL',
                'mobs'  => 'Anges corrompus, créatures saintes hostiles.',
                'dungeons' => '',
                'notes' => 'Zone emblématique HL, excellente pour XP de fin de progression.'
            ],
            [
                'name'  => 'Goddard Castle Town',
                'level' => '70+',
                'type'  => 'Ville HL',
                'mobs'  => '',
                'dungeons' => 'Accès vers les zones volcaniques et les donjons de glace.',
                'notes' => 'Ville orientée contenu haut niveau et raids.'
            ],
        ],
    ],
    [
        'id' => 'schuttgart',
        'title' => 'Région de Schuttgart',
        'description' => 'Région enneigée et industrielle avec plusieurs zones spécifiques.',
        'image' => 'assets/maps/schuttgart_region.jpg',
        'zones' => [
            [
                'name'  => 'Schuttgart Town',
                'level' => '70+',
                'type'  => 'Ville HL',
                'mobs'  => '',
                'dungeons' => 'Accès à Pavel Ruins, plaines enneigées…',
                'notes' => 'Ville optionnelle pour varier les zones HL.'
            ],
            [
                'name'  => 'Pavel Ruins',
                'level' => '70 – 78',
                'type'  => 'Zone / donjon',
                'mobs'  => 'Golems, créatures mécaniques, expérimentations ratées.',
                'dungeons' => 'Complexe industriel semi-donjon.',
                'notes' => 'Zone originale pour changer des spots classiques.'
            ],
            [
                'name'  => 'Frozen Plains',
                'level' => '70 – 80',
                'type'  => 'Zone d\'XP HL',
                'mobs'  => 'Monstres de glace, bêtes polaires…',
                'dungeons' => '',
                'notes' => 'Zone HL pour ceux qui aiment l’ambiance neige / glace.'
            ],
        ],
    ],
    [
        'id' => 'gracia',
        'title' => 'Gracia & zones aériennes',
        'description' => 'Continent de Gracia et contenus spécifiques (selon version du serveur).',
        'image' => 'assets/maps/gracia_region.jpg',
        'zones' => [
            [
                'name'  => 'Gracia Base',
                'level' => '76+',
                'type'  => 'Zone HL',
                'mobs'  => 'Soldats, créatures mutées…',
                'dungeons' => 'Accès aux zones instanciées et aériennes.',
                'notes' => 'Contenu variable selon la chronicle exacte de ton serveur.'
            ],
            [
                'name'  => 'Seed of Destruction',
                'level' => 'Raid HL',
                'type'  => 'Instance / raid',
                'mobs'  => 'Boss et monstres spécifiques à la Seed.',
                'dungeons' => 'Instance de raid majeure.',
                'notes' => 'Contenu organisé, souvent joué en clan / alliance.'
            ],
            [
                'name'  => 'Seed of Infinity',
                'level' => 'Raid HL',
                'type'  => 'Instance / raid',
                'mobs'  => 'Créatures d’ombre, boss de fin de contenu.',
                'dungeons' => 'Instance HL, nécessite une grosse organisation.',
                'notes' => 'Destinée au contenu end-game hautement organisé.'
            ],
        ],
    ],
    [
        'id' => 'hellbound',
        'title' => 'Hellbound',
        'description' => 'Zone spéciale très haut niveau, débloquée selon la progression serveur.',
        'image' => 'assets/maps/hellbound.jpg',
        'zones' => [
            [
                'name'  => 'Hellbound Island',
                'level' => '78+',
                'type'  => 'Zone HL spéciale',
                'mobs'  => 'Monstres démoniaques, créatures corrompues.',
                'dungeons' => 'Plusieurs sous-zones avec mécaniques particulières.',
                'notes' => 'Zone à progression, souvent liée à des quêtes serveur.'
            ],
        ],
    ],
];

include 'includes/header.php';
?>

<style>
    .map-tabs {
        display: flex;
        flex-wrap: wrap;
        gap: 0.5rem;
        margin-bottom: 1rem;
        align-items: center;
    }
    .map-tab {
        padding: 0.3rem 0.9rem;
        border-radius: 999px;
        border: 1px solid #1f2937;
        background: #020617;
        font-size: 0.82rem;
        color: #e5e7eb;
        cursor: pointer;
        text-transform: uppercase;
        letter-spacing: 0.06em;
        box-shadow: 0 6px 12px rgba(15,23,42,0.8);
        transition: all 0.18s ease-out;
    }
    .map-tab:hover {
        border-color: #3b82f6;
        background: radial-gradient(circle at top, rgba(56,189,248,0.25), #020617);
        box-shadow: 0 0 16px rgba(56,189,248,0.4);
    }
    .map-tab.active {
        border-color: #3b82f6;
        background: linear-gradient(135deg, #38bdf8, #1d4ed8);
        color: #020617;
        font-weight: 600;
        box-shadow: 0 0 18px rgba(56,189,248,0.8);
    }

    /* Bouton spécial Carte des PNJ – même taille, style plus sobre */
    .map-tab-pnj {
        padding: 0.3rem 0.9rem;
        border-radius: 999px;
        border: 1px solid #22c55e;
        background: rgba(34,197,94,0.08);
        font-size: 0.82rem;
        color: #bbf7d0;
        text-transform: uppercase;
        letter-spacing: 0.06em;
        box-shadow: 0 6px 12px rgba(15,23,42,0.8);
        text-decoration: none;
        font-weight: 600;
        margin-left: 0.3rem;
        white-space: nowrap;
        transition: all 0.18s ease-out;
    }
    .map-tab-pnj:hover {
        background: linear-gradient(135deg,#22c55e,#16a34a);
        color: #022c22;
        border-color: #22c55e;
        box-shadow: 0 0 18px rgba(34,197,94,0.7);
        text-decoration: none;
        transform: translateY(-1px);
    }

    .map-panel {
        display: none;
    }
    .map-panel.active {
        display: block;
    }
    .map-image-wrapper {
        border-radius: 1rem;
        overflow: hidden;
        border: 1px solid #1f2937;
        margin-top: 0.9rem;
        margin-bottom: 0.9rem;
        background: #020617;
    }
    /* Afficher la carte entière, sans la couper */
    .map-image-wrapper img {
        display: block;
        width: 100%;
        height: auto;
        object-fit: contain;
    }
    .zone-list {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(220px, 1fr));
        gap: 0.7rem;
        margin-top: 0.4rem;
    }
    .zone-card {
        border-radius: 0.9rem;
        border: 1px solid #1f2937;
        background: rgba(15,23,42,0.98);
        padding: 0.6rem 0.7rem;
        font-size: 0.82rem;
    }
    .zone-name {
        font-weight: 600;
        margin-bottom: 0.15rem;
        font-size: 0.88rem;
    }
    .zone-meta {
        font-size: 0.76rem;
        color: #9ca3af;
        margin-bottom: 0.25rem;
    }
    .zone-section-title {
        font-size: 0.78rem;
        font-weight: 600;
        margin-top: 0.3rem;
        margin-bottom: 0.1rem;
        color: #93c5fd;
    }
    .zone-text {
        font-size: 0.8rem;
        line-height: 1.4;
    }
    .map-search-input {
        flex: 1 1 220px;
        padding: 0.4rem 0.6rem;
        border-radius: 999px;
        border: 1px solid #1f2937;
        background: #020617;
        color: #e5e7eb;
        font-size: 0.85rem;
    }
    .map-select {
        flex: 1 1 200px;
        padding: 0.4rem 0.6rem;
        border-radius: 999px;
        border: 1px solid #1f2937;
        background: #020617;
        color: #e5e7eb;
        font-size: 0.85rem;
    }
</style>

<div class="card">
    <h1 class="page-title">Cartes & zones</h1>
    <p class="page-intro">
        Consulte les différentes cartes du monde de <?= htmlspecialchars($serverName) ?>,
        avec les zones d’XP, les donjons et les secteurs dangereux.
    </p>

    <?php if (empty($maps)): ?>
        <p class="muted">Aucune carte n’est configurée pour le moment.</p>
    <?php else: ?>
        <!-- Onglets de maps + bouton Carte des PNJ à la suite -->
        <div class="map-tabs" id="mapTabs">
            <?php foreach ($maps as $index => $map): ?>
                <button
                    type="button"
                    class="map-tab<?= ($index === 0 ? ' active' : '') ?>"
                    data-map-id="<?= htmlspecialchars($map['id']) ?>"
                >
                    <?= htmlspecialchars($map['title']) ?>
                </button>
            <?php endforeach; ?>

            <!-- Bouton spécial Carte des PNJ, même ligne, style plus sobre -->
            <a href="pnj_map.php" class="map-tab-pnj">Carte des PNJ</a>
        </div>

        <!-- Zone de recherche + menu déroulant -->
        <form onsubmit="return false;" style="margin:0.2rem 0 0.8rem 0;display:flex;gap:0.5rem;flex-wrap:wrap;align-items:center;">
            <input
                type="text"
                id="zoneSearchInput"
                class="map-search-input"
                placeholder="Rechercher une zone (nom, type, mobs, donjons)..."
            >

            <select id="zoneSelect" class="map-select">
                <option value="">Toutes les zones</option>
                <?php foreach ($maps as $map): ?>
                    <?php foreach ($map['zones'] as $zone): ?>
                        <?php
                        if (!isset($zone['name']) || $zone['name'] === '') {
                            continue;
                        }
                        ?>
                        <option
                            value="<?= htmlspecialchars($zone['name']) ?>"
                            data-map-id="<?= htmlspecialchars($map['id']) ?>"
                        >
                            <?= htmlspecialchars($zone['name']) ?> (<?= htmlspecialchars($map['title']) ?>)
                        </option>
                    <?php endforeach; ?>
                <?php endforeach; ?>
            </select>
        </form>

        <!-- Panneaux de chaque carte -->
        <?php foreach ($maps as $index => $map): ?>
            <?php $panelId = 'map-panel-' . $map['id']; ?>
            <div
                class="map-panel<?= ($index === 0 ? ' active' : '') ?>"
                id="<?= htmlspecialchars($panelId) ?>"
            >
                <p class="muted" style="margin-bottom:0.6rem;">
                    <?= htmlspecialchars($map['description']) ?>
                </p>

                <?php if (empty($map['zones'])): ?>
                    <p class="muted">Aucune zone n’est encore définie pour cette carte.</p>
                <?php else: ?>
                    <!-- ZONES EN HAUT -->
                    <div class="zone-list">
                        <?php foreach ($map['zones'] as $zone): ?>
                            <?php
                            $name  = isset($zone['name']) ? $zone['name'] : '';
                            $level = isset($zone['level']) ? $zone['level'] : '';
                            $type  = isset($zone['type']) ? $zone['type'] : '';
                            $mobs  = isset($zone['mobs']) ? $zone['mobs'] : '';
                            $dung  = isset($zone['dungeons']) ? $zone['dungeons'] : '';
                            $notes = isset($zone['notes']) ? $zone['notes'] : '';

                            $searchIndex = strtolower($name . ' ' . $level . ' ' . $type . ' ' . $mobs . ' ' . $dung . ' ' . $notes);
                            ?>
                            <article
                                class="zone-card"
                                data-map-id="<?= htmlspecialchars($map['id']) ?>"
                                data-search="<?= htmlspecialchars($searchIndex) ?>"
                            >
                                <div class="zone-name">
                                    <?= htmlspecialchars($name !== '' ? $name : 'Zone sans nom') ?>
                                </div>
                                <div class="zone-meta">
                                    <?php if ($type !== ''): ?>
                                        <?= htmlspecialchars($type) ?>
                                    <?php endif; ?>
                                    <?php if ($type !== '' && $level !== ''): ?>
                                        ·
                                    <?php endif; ?>
                                    <?php if ($level !== ''): ?>
                                        Niveaux <?= htmlspecialchars($level) ?>
                                    <?php endif; ?>
                                </div>

                                <?php if ($mobs !== ''): ?>
                                    <div class="zone-section-title">Monstres</div>
                                    <div class="zone-text"><?= nl2br(htmlspecialchars($mobs)) ?></div>
                                <?php endif; ?>

                                <?php if ($dung !== ''): ?>
                                    <div class="zone-section-title">Donjons / instances</div>
                                    <div class="zone-text"><?= nl2br(htmlspecialchars($dung)) ?></div>
                                <?php endif; ?>

                                <?php if ($notes !== ''): ?>
                                    <div class="zone-section-title">Notes</div>
                                    <div class="zone-text"><?= nl2br(htmlspecialchars($notes)) ?></div>
                                <?php endif; ?>
                            </article>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>

                <?php if (!empty($map['image'])): ?>
                    <!-- CARTE EN BAS -->
                    <div class="map-image-wrapper">
                        <img src="<?= htmlspecialchars($map['image']) ?>"
                             alt="Carte : <?= htmlspecialchars($map['title']) ?>">
                    </div>
                <?php endif; ?>
            </div>
        <?php endforeach; ?>
    <?php endif; ?>
</div>

<script>
(function () {
    var tabs = document.querySelectorAll('.map-tab');
    var panels = document.querySelectorAll('.map-panel');
    var searchInput = document.getElementById('zoneSearchInput');
    var zoneSelect = document.getElementById('zoneSelect');
    var zoneCards = document.querySelectorAll('.zone-card');

    function setActiveMap(mapId) {
        for (var i = 0; i < tabs.length; i++) {
            var t = tabs[i];
            t.classList.toggle('active', t.getAttribute('data-map-id') === mapId);
        }
        for (var j = 0; j < panels.length; j++) {
            var p = panels[j];
            p.classList.toggle('active', p.id === 'map-panel-' + mapId);
        }
        applyZoneFilter();
    }

    function getActiveMapId() {
        for (var i = 0; i < tabs.length; i++) {
            if (tabs[i].classList.contains('active')) {
                return tabs[i].getAttribute('data-map-id');
            }
        }
        return null;
    }

    function applyZoneFilter() {
        var term = '';
        if (searchInput && searchInput.value) {
            term = searchInput.value.toLowerCase();
        }
        var activeMapId = getActiveMapId();

        for (var i = 0; i < zoneCards.length; i++) {
            var card = zoneCards[i];
            var cardMapId = card.getAttribute('data-map-id');
            var text = card.getAttribute('data-search') || '';

            var belongsToActiveMap = (cardMapId === activeMapId);
            var matchesSearch = (term === '' || text.indexOf(term) !== -1);

            if (belongsToActiveMap && matchesSearch) {
                card.style.display = '';
            } else {
                card.style.display = 'none';
            }
        }
    }

    for (var i = 0; i < tabs.length; i++) {
        (function (tab) {
            tab.addEventListener('click', function () {
                setActiveMap(tab.getAttribute('data-map-id'));
                applyZoneFilter();
            });
        })(tabs[i]);
    }

    if (searchInput) {
        searchInput.addEventListener('input', function () {
            if (zoneSelect) {
                zoneSelect.value = '';
            }
            applyZoneFilter();
        });
    }

    if (zoneSelect) {
        zoneSelect.addEventListener('change', function () {
            var option = zoneSelect.options[zoneSelect.selectedIndex];
            var zoneName = option.value;
            var mapId = option.getAttribute('data-map-id');

            if (!zoneName) {
                if (searchInput) {
                    searchInput.value = '';
                }
                applyZoneFilter();
                return;
            }

            if (mapId) {
                setActiveMap(mapId);
            }

            if (searchInput) {
                searchInput.value = zoneName;
            }
            applyZoneFilter();

            var firstVisible = null;
            for (var i = 0; i < zoneCards.length; i++) {
                if (zoneCards[i].style.display !== 'none') {
                    firstVisible = zoneCards[i];
                    break;
                }
            }
            if (firstVisible && firstVisible.scrollIntoView) {
                firstVisible.scrollIntoView({ behavior: 'smooth', block: 'start' });
            }
        });
    }

    if (tabs.length > 0) {
        setActiveMap(tabs[0].getAttribute('data-map-id'));
    }
})();
</script>

<?php include 'includes/footer.php'; ?>

