<?php
$pageTitle = 'Comptes LoginServer';
require_once 'config.php';

$ls = getLsPdo();   // Base LoginServer (l2srls)
$gs = getGsPdo();   // Base GameServer  (l2srgs)

$accounts = [];
$error    = null;

try {
    // 1) Récupération de la liste des logins côté LoginServer
    $stmt = $ls->query("
        SELECT login
        FROM accounts
        ORDER BY login ASC
    ");
    $logins = $stmt->fetchAll(PDO::FETCH_COLUMN);

    if (!empty($logins)) {
        // Initialiser le tableau des comptes avec les colonnes supplémentaires vides
        foreach ($logins as $login) {
            $accounts[] = [
                'login'              => $login,
                'char_name'          => null,
                'clan_name'          => null,
                'lastAccessFormatted'=> null,
            ];
        }

        // 2) Récupérer les personnages associés dans la base GameServer
        //    On prend l'info perso + clan + dernière connexion.
        $placeholders = implode(',', array_fill(0, count($logins), '?'));

        $sqlChars = "
            SELECT
                c.account_name,
                c.char_name,
                c.lastAccess,
                cl.clan_name
            FROM characters AS c
            LEFT JOIN clan_data AS cl ON cl.clan_id = c.clanid
            WHERE c.account_name IN ($placeholders)
            ORDER BY c.level DESC, c.lastAccess DESC
        ";

        $stmtChars = $gs->prepare($sqlChars);
        $stmtChars->execute($logins);

        $charsByAccount = [];

        while ($row = $stmtChars->fetch(PDO::FETCH_ASSOC)) {
            $accName = $row['account_name'] ?? '';
            if ($accName === '') {
                continue;
            }

            // Normalisation en minuscule pour éviter les soucis de casse
            // (Elerandiel vs elerandiel par ex.)
            $key = strtolower($accName);

            // On ne garde que le premier résultat par compte (considéré comme "perso principal")
            if (!isset($charsByAccount[$key])) {
                $last      = $row['lastAccess'] ?? null;
                $formatted = null;

                if (!empty($last) && $last != 0) {
                    $ts = (int)$last;

                    // Beaucoup de L2J stockent en millisecondes -> on adapte
                    if ($ts > 2000000000) { // trop grand pour des secondes
                        $ts = (int)($ts / 1000);
                    }
                    if ($ts > 0) {
                        $formatted = date('Y-m-d H:i', $ts);
                    }
                }

                $charsByAccount[$key] = [
                    'char_name'           => $row['char_name'] ?? null,
                    'clan_name'           => $row['clan_name'] ?? null,
                    'lastAccessFormatted' => $formatted,
                ];
            }
        }

        // 3) Fusionner les infos perso/clan/date dans le tableau des comptes
        foreach ($accounts as &$acc) {
            $key  = strtolower($acc['login']);      // même normalisation ici
            $info = $charsByAccount[$key] ?? null;

            $acc['char_name']           = $info['char_name'] ?? null;
            $acc['clan_name']           = $info['clan_name'] ?? null;
            $acc['lastAccessFormatted'] = $info['lastAccessFormatted'] ?? null;
        }
        unset($acc);
    }
} catch (Throwable $e) {
    $error = $e->getMessage();
}

include 'includes/header.php';
?>

<div class="card">
    <h1 class="page-title">Comptes LoginServer</h1>
    <p class="page-intro">
        Liste des comptes enregistrés côté LoginServer. Pratique pour suivre l’activité globale de
        <?= htmlspecialchars($SERVER_NAME) ?>.
    </p>

    <?php if ($error): ?>
        <p style="color:#f97373;">Erreur lors de la lecture des comptes : <?= htmlspecialchars($error) ?></p>
    <?php elseif (empty($accounts)): ?>
        <p class="muted">Aucun compte trouvé pour le moment.</p>
    <?php else: ?>
        <table style="width:100%;border-collapse:collapse;font-size:0.9rem;margin-top:0.5rem;">
            <thead>
                <tr style="border-bottom:1px solid #1f2937;">
                    <th style="text-align:left;padding:0.35rem 0.4rem;color:#9ca3af;">Login</th>
                    <th style="text-align:left;padding:0.35rem 0.4rem;color:#9ca3af;">Personnage principal</th>
                    <th style="text-align:left;padding:0.35rem 0.4rem;color:#9ca3af;">Guilde</th>
                    <th style="text-align:left;padding:0.35rem 0.4rem;color:#9ca3af;">Dernière connexion</th>
                </tr>
            </thead>
            <tbody>
            <?php foreach ($accounts as $acc): ?>
                <tr style="border-bottom:1px solid #111827;">
                    <td style="padding:0.3rem 0.4rem;"><?= htmlspecialchars($acc['login']) ?></td>
                    <td style="padding:0.3rem 0.4rem;">
                        <?= $acc['char_name'] ? htmlspecialchars($acc['char_name']) : '—' ?>
                    </td>
                    <td style="padding:0.3rem 0.4rem;">
                        <?= $acc['clan_name'] ? htmlspecialchars($acc['clan_name']) : '—' ?>
                    </td>
                    <td style="padding:0.3rem 0.4rem;">
                        <?= $acc['lastAccessFormatted'] ? htmlspecialchars($acc['lastAccessFormatted']) : '—' ?>
                    </td>
                </tr>
            <?php endforeach; ?>
            </tbody>
        </table>
    <?php endif; ?>
</div>

<?php include 'includes/footer.php'; ?>
