<?php
// items.php – Liste des objets du serveur (armes, armures, bijoux, autres)

$pageTitle = 'Objets';
require_once 'config.php';

$gs = getGsPdo(); // base GameServer

$search = trim($_GET['q'] ?? '');
$type   = $_GET['type'] ?? 'all';   // all, Arme, Armure, Bijou, Objet
$grade  = trim($_GET['grade'] ?? ''); // S, A, B, C, D...

$items = [];
$error = null;

try {
    // On part sur la structure L2J classique : weapon / armor / etcitem.
    // Si tes tables ont d'autres noms, on adaptera ensuite.
    $sql = "
        SELECT item_id, name, type_name, crystal_type, bodypart
        FROM (
            SELECT item_id, name, 'Arme' AS type_name, crystal_type, bodypart
            FROM weapon
            UNION ALL
            SELECT item_id, name, 'Armure' AS type_name, crystal_type, bodypart
            FROM armor
            UNION ALL
            -- Bijoux (etcitem avec bodypart de bijoux)
            SELECT item_id, name, 'Bijou' AS type_name, crystal_type, bodypart
            FROM etcitem
            WHERE bodypart IN ('neck','rear','lear','rfinger','lfinger')
            UNION ALL
            -- Autres objets
            SELECT item_id, name, 'Objet' AS type_name, crystal_type, bodypart
            FROM etcitem
            WHERE bodypart NOT IN ('neck','rear','lear','rfinger','lfinger')
        ) AS items
        WHERE 1
    ";

    $bind = [];

    if ($type !== 'all') {
        $sql .= " AND type_name = :type_name";
        $bind[':type_name'] = $type;
    }

    if ($grade !== '') {
        $sql .= " AND crystal_type = :grade";
        $bind[':grade'] = strtoupper($grade);
    }

    if ($search !== '') {
        $sql .= " AND (name LIKE :term";
        $bind[':term'] = '%' . $search . '%';

        if (ctype_digit($search)) {
            $sql .= " OR item_id = :idexact";
            $bind[':idexact'] = (int)$search;
        }

        $sql .= ")";
    }

    $sql .= " ORDER BY type_name, crystal_type DESC, name ASC LIMIT 500";

    $stmt = $gs->prepare($sql);
    $stmt->execute($bind);
    $items = $stmt->fetchAll(PDO::FETCH_ASSOC);

} catch (Throwable $e) {
    $error = "Une erreur est survenue lors du chargement des objets.";
}

include 'includes/header.php';
?>

<div class="card">
    <h1 class="page-title">Objets du serveur</h1>
    <p class="page-intro">
        Liste des armes, armures, bijoux et autres objets de <?= htmlspecialchars($SERVER_NAME ?? 'Thanys') ?>.
        Utilise les filtres pour trouver rapidement ce que tu cherches.
    </p>

    <?php if ($error): ?>
        <p style="color:#f97373;"><?= htmlspecialchars($error) ?></p>
    <?php endif; ?>

    <!-- Filtres / recherche -->
    <form method="get" style="margin:0.8rem 0;display:flex;gap:0.5rem;flex-wrap:wrap;align-items:center;">
        <input
            type="text"
            name="q"
            value="<?= htmlspecialchars($search) ?>"
            placeholder="Rechercher par nom ou ID d'objet..."
            style="
                flex:2 1 220px;
                padding:0.4rem 0.6rem;
                border-radius:999px;
                border:1px solid #1f2937;
                background:#020617;
                color:#e5e7eb;
                font-size:0.85rem;
            "
        >

        <select name="type"
            style="
                flex:1 1 120px;
                padding:0.4rem 0.6rem;
                border-radius:999px;
                border:1px solid #1f2937;
                background:#020617;
                color:#e5e7eb;
                font-size:0.85rem;
            ">
            <option value="all"  <?= $type === 'all' ? 'selected' : '' ?>>Tous les types</option>
            <option value="Arme" <?= $type === 'Arme' ? 'selected' : '' ?>>Armes</option>
            <option value="Armure" <?= $type === 'Armure' ? 'selected' : '' ?>>Armures</option>
            <option value="Bijou" <?= $type === 'Bijou' ? 'selected' : '' ?>>Bijoux</option>
            <option value="Objet" <?= $type === 'Objet' ? 'selected' : '' ?>>Autres objets</option>
        </select>

        <select name="grade"
            style="
                flex:0 0 90px;
                padding:0.4rem 0.6rem;
                border-radius:999px;
                border:1px solid #1f2937;
                background:#020617;
                color:#e5e7eb;
                font-size:0.85rem;
            ">
            <option value=""  <?= $grade === '' ? 'selected' : '' ?>>Grade</option>
            <option value="S"  <?= strtoupper($grade) === 'S' ? 'selected' : '' ?>>S</option>
            <option value="A"  <?= strtoupper($grade) === 'A' ? 'selected' : '' ?>>A</option>
            <option value="B"  <?= strtoupper($grade) === 'B' ? 'selected' : '' ?>>B</option>
            <option value="C"  <?= strtoupper($grade) === 'C' ? 'selected' : '' ?>>C</option>
            <option value="D"  <?= strtoupper($grade) === 'D' ? 'selected' : '' ?>>D</option>
        </select>

        <button type="submit"
            style="
                padding:0.4rem 0.9rem;
                border-radius:999px;
                border:none;
                cursor:pointer;
                font-size:0.85rem;
                font-weight:600;
                background:linear-gradient(135deg,#38bdf8,#1d4ed8);
                color:#020617;
            ">
            Rechercher
        </button>
    </form>

    <!-- Résultats -->
    <?php if (empty($items)): ?>
        <p class="muted">Aucun objet trouvé avec ces critères.</p>
    <?php else: ?>
        <div style="overflow:auto;margin-top:0.5rem;">
            <table style="width:100%;border-collapse:collapse;font-size:0.84rem;">
                <thead>
                    <tr style="border-bottom:1px solid #1f2937;">
                        <th style="text-align:left;padding:0.35rem 0.4rem;color:#9ca3af;">ID</th>
                        <th style="text-align:left;padding:0.35rem 0.4rem;color:#9ca3af;">Nom</th>
                        <th style="text-align:left;padding:0.35rem 0.4rem;color:#9ca3af;">Type</th>
                        <th style="text-align:left;padding:0.35rem 0.4rem;color:#9ca3af;">Grade</th>
                        <th style="text-align:left;padding:0.35rem 0.4rem;color:#9ca3af;">Emplacement</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($items as $it): ?>
                        <tr style="border-bottom:1px solid #111827;">
                            <td style="padding:0.3rem 0.4rem;"><?= (int)$it['item_id'] ?></td>
                            <td style="padding:0.3rem 0.4rem;"><?= htmlspecialchars($it['name']) ?></td>
                            <td style="padding:0.3rem 0.4rem;"><?= htmlspecialchars($it['type_name']) ?></td>
                            <td style="padding:0.3rem 0.4rem;"><?= htmlspecialchars($it['crystal_type']) ?></td>
                            <td style="padding:0.3rem 0.4rem;"><?= htmlspecialchars($it['bodypart']) ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            <p class="muted" style="margin-top:0.4rem;">
                Affichage limité aux 500 premiers résultats. Utilise les filtres pour affiner.
            </p>
        </div>
    <?php endif; ?>
</div>

<?php include 'includes/footer.php'; ?>


